/*
  Ball for Bombing Run game. Must get ball and take it to enemy goal.
  Author Mark Caldwell aka W@rHe@d of The Reliquary
    
  Generally refered to as the Ball because in it's most basic sense, it's a ball.
  Also called the Orb or the Bomb. 
  Called the Orb during game play because UT does not have any 'ball' voice messages, 
  so we just use the warfare orb voice messages.
  We also use the warfare orb graphics.
  If we had 'ball' voice messages to use, then we would not have to refer to it as an orb at all.
  We could use other graphics besides the orb graphics if we wished.
  It's also a bomb because the basic concept of the game is that you're carrying the bomb to the enemy 
  base to blow it up.
*/

//we extend UTOnslaughtFlag to make the "You can not enter a vehicle carrying the orb" message happen.
//it's hard coded in UTPlayerController.CheckPickedVehicle to be 'Flag' or 'Orb' based on class type.
class UTBRBall extends UTOnslaughtFlag;

//used by DetachFromHolder
const DETACH_DROP = 1;   
const DETACH_THROW = 2; 
const DETACH_SEND_HOME = 3;   

var StaticMeshComponent BallMesh;
var CylinderComponent CylinderComponent;
var RepNotify bool isDropped;
var float CustomGravityScaling;
var RepNotify int TeamIndex;
var Pawn LastHolder;
var UTPlayerReplicationInfo LastHolderPRI;  //needed because if shooter dies while scoring their pawn's player replication info is lost
var UTBRPlayerManager HolderPlayerManager;
var float Elasticity;
var vector OrigAccel;
var float AccelRate;
var float TossZ;
var SoundCue LockAcquiredSound;
var SoundCue LockLostSound;
var() float ThrowerTouchDelay;
var() float SeekInterval;
var() transient float SeekAccum;
var repnotify Pawn PassTarget;
var float Mass;
var SoundCue ImpactSound;
var ParticleSystem BallEffects[3];
var ParticleSystemComponent BallEffectComp;
var Material BallMaterials[3];
var float LastLocationPingTime;
var UTBRBallLauncher holderBallLauncher;
var RepNotify bool Enabled;
var RepNotify bool PassSeeking;
var DynamicLightEnvironmentComponent BallSurfaceLight;
var LinearColor BallSurfaceColors[3];
var PointLightComponent BallLight;
var color BallColors[3];
var Actor EnemyGoal;  //enemy goal for current ball holder
var bool ShotByBallLauncher;  //used for scoring
var bool HasHitWall;
var UTBRCarrierEffect CarrierEffects[2];
var float CameraViewDistance;             /** distance away for camera when viewtarget */
var vector CameraLastLocation;
var int PriorTeamNum;
var UTEmitter flightTrail;
var RepNotify bool isThrown;
var Controller HolderController;
var UTBot TryingToDeflect;
var bool Resetting;
var Vehicle OldHolderVehicle;


simulated function PostBeginPlay()
{    
    Super(UTCarriedObject).PostBeginPlay();   
   
    SetBallMaterials();
    
    SpawnCarrierEffect(0);
    SpawnCarrierEffect(1);

    MaxDropTime = 30.0;
}

//this gay function must be overridden or ball will get reset if bot can't reach ball
event NotReachableBy(Pawn P)
{
}

simulated event SetOrbTeam()
{
}

simulated function DrawIcon(Canvas Canvas, vector IconLocation, float IconWidth, float IconAlpha)
{
   Super(UTCarriedObject).DrawIcon(Canvas, IconLocation, IconWidth, IconAlpha);
}

simulated event PostRenderFor(PlayerController PC, Canvas Canvas, vector CameraPosition, vector CameraDir)
{
    Super(UTCarriedObject).PostRenderFor(PC, Canvas, CameraPosition, CameraDir);
}

function SendFlagMessage(Controller C)
{
    Super(UTCarriedObject).SendFlagMessage(C);
}

simulated function UpdateTeamEffects()
{
}

function SetTeam(int iTeamIndex)
{
}

function BroadcastDroppedMessage(Controller EventInstigator)
{
    Super(UTCarriedObject).BroadcastDroppedMessage(EventInstigator);
}

//Score() is called by a power node when orb touches it and node get rebuilt.
//nothing we can do about this behavior or ball rebuilding node, as it's in the ut class.
function Score()
{
}

//called when goal is scored. we don't use Score() because power nodes call it.
function GoalScore()
{
    GetKismetEventObjective().TriggerFlagEvent('Captured', LastHolder.Controller);

    if ( !WorldInfo.Game.bGameEnded )
    {
        SetEnabled(false);
        SendHome(None);	
    }
}

function Reset()
{
}

simulated function String GetHumanReadableName()
{
	return "Ball";
}

simulated function Actor Position()
{
    if (IsHeld() && (LastHolder != None))
    {
        return LastHolder;
    }
    else
    {
        return self;
    }
}

//return true if ball is moving
function bool IsInMotion()
{
    //don't count very small movement as motion
    return vsize(Position().Velocity) > 100;    
}

simulated function bool IsHeldByTeamOf(Pawn p)
{
    return (P != none) && IsHeld() && (GetTeamNum() == p.GetTeamNum());
}

simulated function bool IsHeldByEnemyOf(Pawn p)
{
    return (P != none) && IsHeld() && (GetTeamNum() != p.GetTeamNum());
}

simulated function bool IsHeldBy(Pawn p)
{   
    return 
    (HolderPRI != none) &&
    (HolderPRI == UTBRGame(WorldInfo.Game).GetPlayerReplicationInfo(p));
}

//return true if possessed by a team, but not necessarily held.
//possession is determined by the last team to touch ball.
//ball has color of possessed team.
//possession ends when ball is returned home.
simulated function bool IsPossessed()
{
    return (GetTeamNum() == 0) || (GetTeamNum() == 1);
}

//return true if p's team has possession of ball.
//ball might be dropped but last holder was a team member,
//usually meaning team member is nearby and is in control of the ball.
simulated function bool IsPossessedByTeamOf(Pawn p)
{
    return (P != none) && (GetTeamNum() == p.GetTeamNum());
}

//return true if ball is under p's team's controll.
//note that if ball is dropped but last holder is still near ball
//then they are most likely still in control of ball. 
//common situation is for runner to drop ball and perpare for a run
//while waiting for attackers to hit base.
simulated function bool IsControlledByTeamOf(Pawn p)
{
    return IsPossessedByTeamOf(p) && 
           (LastHolder != none) &&
           (LastHolder.Health > 0) && 
           (vsize(location - LastHolder.location) < 1500);
}

//return true if ball is controlled by someone on P's team besides P
simulated function bool IsControlledByOtherTeamMember(Pawn p)
{
   return IsControlledByTeamOf(P) && (LastHolder != P);
}

simulated function bool IsHeld()
{
    return HolderPRI != none;
}

//create the rings around ball holder
function SpawnCarrierEffect(int teamNum)
{
   CarrierEffects[teamNum] = spawn(class'UTBRCarrierEffect');
   CarrierEffects[teamNum].SetTeam(teamNum);  
}

//attach the rings around ball holder
function AttachCarrierEffect(Pawn p, int teamNum)
{
    if (p == none)
    {
        return;
    }
    
    if (CarrierEffects[teamNum] == none)
    {
       SpawnCarrierEffect(teamNum);
    }
    
    CarrierEffects[teamNum].AttachTo(p, '');
}

//detach rings from ball holder
function DetachCarrierEffect(int teamNum)
{
    CarrierEffects[teamNum].DetachFromHolder(); 
}

function Controller GetController(Pawn P)
{   
    return UTBRGame(WorldInfo.Game).GetController(P);
}

function BallTimer()
{
    local int newHealth;
      
    if (Holder == none)
    {
        return;
    }  
           
    newHealth = Holder.Health + int(UTBRGame(WorldInfo.Game).Settings.BallCarrierHealthRegenerated);
    if (newHealth > Holder.HealthMax)
    {
        newHealth = Holder.HealthMax;
    }

    if (Holder.Health < newHealth)
    {   
        Holder.Health = newHealth;
    }

    //even though ball launcher's BringUpTimer helps to stop launcher from being switched out,
    //sometimes ball launcher can still get switched out in weird situations. 
    //below call will prevent that.
    //could duplicate by pounding on a numeric key (in net play) when picking up ball. sometimes after doing that
    //even though launcher appeared to be up, when you fired it would switch out to the other weapon instead of firing ball.
    if (holderBallLauncher != none)
    {
        holderBallLauncher.CheckIfActive();
    }
}

//called client side when certain vars are replicated.
//must trap certain events to make sure things replicate properly
//because otherwise they will not replicate properly during net play.
//ut replication seems to have many bugs.
simulated event ReplicatedEvent(name VarName)
{
    if (VarName == 'TeamIndex')
    {
        SetBallMaterials();
    }

    //got many problems replicating certain things, so override client side
     
    if ((VarName == 'HolderPRI') && (HolderPRI != none))
    {
        ClientSetHolder();
    }
            
    if ((VarName == 'isDropped') && isDropped)
    {
        ClientDetachFromHolder();
    }
    
    if ((VarName == 'isThrown') && isThrown)
    {
       ClientThrown();
    }
    
    if (VarName == 'PassSeeking')
    {
        ClientPassSeek();
    }
    
    if (VarName == 'Enabled')
    {
        SetEnabled(Enabled);
    }
    
    Super(UTCarriedObject).ReplicatedEvent(VarName);
}


/** returns true if should be rendered for passed in player */
simulated function bool ShouldMinimapRenderFor(PlayerController PC)
{
    return true;
}

//check and see if new ball position will fit ok into the world and not be within a wall or whatever
function CheckFit()
{
    local vector X,Y,Z;

    GetAxes(OldHolder.Rotation, X,Y,Z);
    SetRotation(rotator(-1 * X));
    if ( !SetLocation(OldHolder.Location - 2 * OldHolder.GetCollisionRadius() * X + OldHolder.GetCollisionHeight() * vect(0,0,0.5))
        && !SetLocation(OldHolder.Location) && (OldHolder.GetCollisionRadius() > 0) )
    {
        //SetCollisionSize(FMin(DefaultRadius,0.8 * OldHolder.GetCollisionRadius()), FMin(DefaultHeight, 0.8 * OldHolder.GetCollisionHeight()));
        if ( !SetLocation(OldHolder.Location) )
        {
            //`log(self$" Drop sent flag home",,'Error');
            AutoSendHome();
            return;
        }
    }
}

//if ball falls into lava and etc, it's an automatic send home
function CheckPain()
{
    if (IsInPain())
    {
        AutoSendHome();
    }
}

//set the yellow/red/blue graphics on ball  
simulated function SetBallMaterials()
{
    local int ix;

    
    if (TeamIndex == -1)
    {
        ix = 2;   
    } 
    else if (TeamIndex >= 0)
    {
        ix = TeamIndex;     
    }
    
    BallMesh.SetMaterial(1,BallMaterials[ix]);
    BallEffectComp.SetTemplate(BallEffects[ix]);
    BallEffectComp.SetActive(true);
    
    BallLight.SetLightProperties(,BallColors[ix]);
    BallSurfaceLight.AmbientGlow = BallSurfaceColors[ix];
       
    if ((UTBRGame(WorldInfo.Game) != none) && bool(UTBRGame(WorldInfo.Game).Settings.NecrisTheme))
    {
        BallMesh.SetMaterial(1, Material'UN_Liquid.SM.Materials.M_UN_Liquid_SM_NanoBlack_03_Master');
    }
}

//send messages that someone has taken the ball
function LogTaken(Controller EventInstigator)
{
    Super(UTCarriedObject).LogTaken(EventInstigator);
    
    if (GetTeamNum() != PriorTeamNum)
    {
        BroadcastTakenDroppedMessage(EventInstigator);
    }      
}

//needed because in UTCarriedObject the auto state is Home, which ignores SendHome
auto state defaultstate
{
}

//called on server and also on client for SendHome() to make sure things replicate ok to client
simulated function ClientReturnedHome()
{
    // things can go a little screwy with replication ordering due to all the base/physics/location changes that go on
    // so we use this to make sure the flag's where it should be clientside
    if (HomeBase != None)
    {
        SetOwner(none);    
        SetOwnerNoSee(false);
        
        bUseTeamColorForIcon = false; 
        SetPhysics(PHYS_None);      
        Velocity = vect(0,0,0);     
        SetCollision(true, false);      
        SetBase(HomeBase);      
        SetRotation(HomeBase.Rotation);     
        
        //this is a better way than using SetLocation. SetLocation has issues in net play.
        //for example if the base was on a moving platform, in net play the ball would stay with the base.  
        SetRelativeLocation(HomeBaseOffset >> HomeBase.Rotation);
    }
}

//send ball to it's home base
function SendHome(Controller Returner)
{
    local bool origEnabled;
    local UTBRPlayerManager pm;
       
    bForceNetUpdate = True;     
        
    if (HomeBase == none)
    {
        destroy();
    }    

    //need to disable ball while sending home, as a touch event can get fired and ball picked up before it's at home.
    //also, even if at home we don't want the touch to get fired until we are all done setting it up at home.
    origEnabled = Enabled;
    SetEnabled(false);
    
    if (IsHeld())
    {
        DetachFromHolder(DETACH_SEND_HOME);
    }

    Resetting = false;
    ShotByBallLauncher = false;
    GotoState('Home');
    SetPhysics(PHYS_NONE);
    Velocity = vect(0,0,0);
    SetReturnTimer(false);
    isDropped = false;
    isThrown = false;
    bHome = true;
    TeamIndex = -1;
    
    //needed to render ball on minimap. See UTMapInfo class.
    //will render in gold color because bUseTeamColorForIcon is false.
    UTBRGame(WorldInfo.Game).Teams[0].TeamFlag = self;        
    UTBRGame(WorldInfo.Game).Teams[1].TeamFlag = none;  
    
    SetBallMaterials();
    CalcSetHome();
    if (returner != none)
    {
        LogReturned(Returner);
    }
    PlaySound(ReturnedSound);
    
    bHome = true;       

    if ( Team != None )
    {
        // note team is none at start of match, but flagstate is already correctly set
        UTGameReplicationInfo(WorldInfo.GRI).SetFlagHome(Team.TeamIndex);
    }

    HomeBase.bForceNetUpdate = TRUE;
    bForceNetUpdate = TRUE;
    
    HomeBase.DefenderTeamIndex = 255;
    Team = none;

    ClientReturnedHome();
    
    SetEnabled(origEnabled);
    
    if (LastHolderPRI != none)
    {
        pm = UTBRGame(WorldInfo.Game).GetPlayerManager(LastHolderPRI);
        if (pm != none)
        {
            pm.ResetBallTime = 0;
        }
    }
       
    UTBRGame(WorldInfo.Game).SetPlayerObjectives(false);
    
    //now it's all set up, can check for things which were standing on the home base
    CheckTouching();
}

state Home
{
	function Reset()
	{
	}
	
    function LogTaken(Controller EventInstigator)
    {
        Global.LogTaken(EventInstigator);
    }

    function BeginState(Name PreviousStateName)
    {
    }
    
    function EndState(Name NextStateName)
    {
    }
}

//called on server and also on client for SetHolder() to make sure things are replicated properly
simulated function ClientSetHolder()
{
        //called when ball is picked up. must perform all of below to make
        //sure ball gets attached properly to holder. 
        //seen all kinds of problems like ball floating around and etc.
        
        if (flightTrail != none)
        {
            flightTrail.Destroy();
            flightTrail = none;
        }         
        
        bUseTeamColorForIcon = true;   
       
        ReleasePassTarget();

        SetPhysics(PHYS_None);
                        
        //these two lines are needed to make sure the SetBase call in HoldGameObject works properly,
        //otherwise the hard attach will not work ok.
        bCollideWorld = false;
        SetCollision(false, false);  
        
        if ( UTPawn(LastHolder) != None )
        {
            SetOwner(LastHolder);          
            UTPawn(LastHolder).HoldGameObject(self);
        }   
    
        //adjust light down so it will still shine on floor    
        BallLight.SetTranslation(vect(0,0, -50));   
}

function AddAssist(Controller C)
{
    local int i;
    
    for (i=0;i<Assists.Length;i++)
    {
        if (Assists[i] == C)
        {
            return;
        }
    }

    Assists[Assists.Length] = C;
}

//set the ball holder to C's pawn
function SetHolder(Controller C)
{
    local UTBRSquadAI S;
    local bool landedOnBall;
   
    Resetting = false;          
    PriorTeamNum = GetTeamNum();
    isDropped = false;
    isThrown = false;
    bForceNetUpdate = True;
    
    if ((UTBot(C) != none) && (C.Pawn != none) && (C.Pawn.Physics == PHYS_Falling) && (vsize(velocity) < 100))
    {
        //ut3 fires a jump when bot lands on sitting ball so this will help control where it jumps to,
        //seeing as how generally we want to land rather than jump.
        landedOnBall = true;
    } 
    
    GotoState('Held');

    HolderController = C;            
    Holder = C.Pawn;
    if (Vehicle(C.Pawn) != none)
    {
        Holder = Vehicle(C.Pawn).Driver;
    }
      
    ShotByBallLauncher = false;
    ReleasePassTarget();
    LastHolder = Holder;
    LastHolderPRI = UTPlayerReplicationInfo(C.PlayerReplicationInfo);
    HolderPlayerManager = UTBRGame(WorldInfo.Game).GetPlayerManager(LastHolderPRI);
    HolderPlayerManager.TheBall = self;
    HolderPlayerManager.LastJumpTime = 0;
    HolderPlayerManager.TriedPath = false;
    HolderPlayerManager.LandedOnBall = landedOnBall;
    
    if (
        (HolderPlayerManager.LastVehicle != none) && (HolderPlayerManager.LastVehicle.Driver == none) &&
        (vsize(HolderPlayerManager.LastVehicle.Location - Holder.Location) < 2000)
       )
    { 
        HolderPlayerManager.LastVehicle.VehicleLostTime = WorldInfo.TimeSeconds;
        HolderPlayerManager.LastVehicle = none;
    }

    AttachCarrierEffect(holder, holder.GetTeamNum());
        
    SetReturnTimer(false);
    bHome = false;
    
    //not good to set this. it makes bots try to defend the ball base.
    //HomeBase.DefenderTeamIndex = C.Pawn.GetTeamNum();
    
    Team = UTBRGame(WorldInfo.Game).Teams[C.Pawn.GetTeamNum()];
    EnemyGoal = UTBRGame(WorldInfo.Game).Goals[1 - C.Pawn.GetTeamNum()];

    //needed to render ball on minimap. See UTMapInfo class.
    UTBRGame(WorldInfo.Game).Teams[C.Pawn.GetTeamNum()].TeamFlag = self;        
    UTBRGame(WorldInfo.Game).Teams[1 - C.Pawn.GetTeamNum()].TeamFlag = none;       
    
    LogTaken(c);  //note: this must be called only after team has been set

    if ( UTBot(C) != None )
    {
        //forces reset of DefensivePosition. now that bot has ball, allow it to route directly to goal
        HolderPlayerManager.LastFormationCenter = none;        
        
        S = UTBRSquadAI(UTBot(C).Squad);
    }
    else if ( PlayerController(C) != None )
    {
        S = UTBRSquadAI(UTTeamInfo(C.PlayerReplicationInfo.Team).AI.FindHumanSquad());
    }

    if ( S != None )
    {
        S.BallTakenByOurTeam(C);
    }   
    
    //in net play with a rapid amount of killing the holder can die while in this routine.
    //it probably happens when BallTakenByOurTeam() is called, which proabably allows some
    //more ut events to fire allowing holder to die.
    //at that point in log there will be Weapon=none errors.
    //so, this check is put above that place.
    if (! ValidHolder(holder))
    {
        Drop();
        return;
    }     

    if (Holder.Weapon != none)
    {
        Holder.Weapon.StopFire(0);      
        Holder.Weapon.StopFire(1);
    }   

    if ( UTPawn(Holder) != None )
    {
        UTPawn(Holder).DeactivateSpawnProtection();
    }
    HolderPRI = LastHolderPRI;
    bHome = false;
    TeamIndex = HolderPRI.Team.TeamIndex;   
    SetBallMaterials();
    HolderPRI.SetFlag(self);
    HolderPRI.bForceNetUpdate = TRUE;
    LastFlagSeeTime = WorldInfo.TimeSeconds - 11;
    UTGameReplicationInfo(WorldInfo.GRI).SetFlagHeldEnemy(GetTeamNum());
    WorldInfo.GRI.bForceNetUpdate = TRUE;  
       
    ClientSetHolder();
    
    bForceNetUpdate = TRUE;
       
    if (PickupSound != None)
    {
        PlaySound(PickupSound);
    }   

    // AI Related
    C.MoveTimer = -1;
    Holder.MakeNoise(2.0);

    // Track First Touch
    if (FirstTouch == None)
        FirstTouch = C;

    AddAssist(C);

    //a hack to make UTVoice.uc play orb message instead of flag message.
    //no support for just passing the orb message we want.
    //UTVoice plays orb message if game class is onslaught.
    WorldInfo.GRI.GameClass = class'UTOnslaughtGame';
    SendFlagMessage(C);  //sends "i've got the orb" message
    
    ForEach Holder.InvManager.InventoryActors( class'UTBRBallLauncher', holderBallLauncher )
    {
        break;
    }
    
    holderBallLauncher.TheBall = self;
    
    if (UTBot(C) != none)
    {       
        if (! bool(UTBRGame(WorldInfo.Game).Settings.AllowBallCarrierWeapons))
        {
            holderPlayerManager.oldImpactJumpZ = UTBot(C).ImpactJumpZ;      
            holderPlayerManager.OldHasTranslocator = UTBot(C).bHasTranslocator;
                    
            UTBot(C).ImpactJumpZ = 0;

            //unfortunately this does not stop bot from pathing like it has a translocator.
            //certain trans vars must be overridden before every path request in UTBRSquadAI.FindPathToObjective.
            if (UTBot(C).bHasTranslocator)
            {
                UTBot(C).bHasTranslocator = false;
                UTBot(C).bAllowedToTranslocate = false;
            }
        }        
    }
   
    SetTimer(1.0, True, 'BallTimer');   
        
    holderBallLauncher.PendingWeapon = holder.weapon;
    Holder.SetActiveWeapon(holderBallLauncher); 
    holderBallLauncher.SetBringUpTimer();
    
    HolderPlayerManager.BallLauncher = holderBallLauncher;
    HolderPlayerManager.CanResetBall = true;   
    
    UTBRGame(WorldInfo.Game).SetPlayerObjectives(false, UTPlayerController(C));
    
    bForceNetUpdate = True;
    
    //see similar call above    
    if (! ValidHolder(holder))
    {
        Drop();
        return;
    }        
}

state Held 
{
	event OrbUnused()
	{
	}
	
	function Score()
	{        	
        //Score() is called by a power node when orb touches it and node get rebuilt.
        //nothing we can do about this behavior or ball rebuilding node, as it's in the ut class.
        
        AutoSendHome(); 
    }
    	
    function LogTaken(Controller EventInstigator)
    {
        Global.LogTaken(EventInstigator);
    }
    
    function SendHome(Controller Returner)
    {
        Global.SendHome(Returner);
    }
    
    function BeginState(Name PreviousStateName)
    {
        ClearTimer();       
    }

    function EndState(Name NextStateName)
    {
    }
}

//drop ball
function Drop(optional Controller Killer)
{
    local int i;
   
    if ((Holder != none) && (! Holder.IsInState('FeigningDeath')) && (Holder.Health > 0))
    {               
        //teleporter automatically drops orb. for br we want to allow holder to keep ball.
        for ( i=0; i<Holder.Touching.Length; i++ )
        {
            if (Teleporter(Holder.Touching[i]) != none)
            {
                return;
            }
        }
    }  

    DetachFromHolder(DETACH_DROP, Killer);
    
    if ((Killer != none) && (UTPlayerReplicationInfo(Killer.PlayerReplicationInfo) != none))
    {
       UTPlayerReplicationInfo(Killer.PlayerReplicationInfo).IncrementEventStat('EVENT_KILLEDFLAGCARRIER');    
    }
}

simulated function RenderMapIcon(UTMapInfo MP, Canvas Canvas, UTPlayerController PlayerOwner)
{
    Super(UTCarriedObject).RenderMapIcon(MP, Canvas, PlayerOwner);
}

simulated function RenderEnemyMapIcon(UTMapInfo MP, Canvas Canvas, UTPlayerController PlayerOwner, UTGameObjective NearbyObjective)
{
    Super(UTCarriedObject).RenderMapIcon(MP, Canvas,PlayerOwner);
}

//choose a bot to defend against the shot ball by trying to boost it away
function ChooseBoostDefender(int teamNum)
{
    local UTBot B;
    local UTBRPlayerManager defender, nearestDefender;
    local float dist, nearestDist, bestDist;
    local bool checkedSkill;
    local UTBRPlayerManager playerManager;
    
    nearestDist = 999999999;
    
    //select best defender.
    //this is the one closest to home goal and is not pointing at goal.
    //don't want a defender who's pointing at goal because they might
    //accidentally boost ball into their goal. 
    
    foreach WorldInfo.AllControllers(class'UTBot', B)
    {
        if (B.GetTeamNum() == teamNum)  
        {
            if (! B.LineOfSightTo(self))
            {
                continue;
            }
            
            playerManager = UTBRGame(WorldInfo.Game).GetPlayerManager(B.PlayerReplicationInfo);
            if (playerManager == none)
            { 
                continue;
            }
            
            if (! playerManager.HasBoostWeapon(false))
            {
                continue;
            }
            
            if (! checkedSkill)
            {
                checkedSkill = true;
                if (! playerManager.ShouldBoostBall())
                {
                    return;
                }
            }
            
            dist = vsize(UTBRSquadAI(B.Squad).FriendlyGoal.Location - B.Pawn.Location);
                       
            if ((dist < bestDist) && B.Pawn.NeedToTurn(UTBRSquadAI(B.Squad).FriendlyGoal.Location)) 
            {
                defender = playerManager;
                bestDist = dist;
            }
            
            if (dist < nearestDist) 
            {
                nearestDist = dist;
                nearestDefender = playerManager;
            }            
        }
    }

    if (defender == none)
    {
        defender = nearestDefender;
    }
        
    if (defender != none)
    {
        defender.StartBoostDefending(self);
    }
}

//called on server and on client for setting up pass seeking to make sure all replicates ok
simulated function ClientPassSeek()
{
    if (PassSeeking)
    {
        SetPhysics(PHYS_PROJECTILE);            
        SetTimer(SeekInterval, true, 'PassSeekTimer');          
    }
    else
    {
        ReleasePassTarget();
    }   
}

function float GetShootSpeed()
{
    local float speed;
    
    speed = float(UTBRGame(WorldInfo.Game).Settings.BallSpeed);
    
    if (Holder != none)
    {
        speed += VSize(Holder.Velocity);
    }
    
    return speed;
}

//return approx distance ball will travel when shot.
//this isn't a mathematically accurate equation, just a rough estimate.
//height is upward Z from current position at distance 1000 bot is aiming up.
function float TravelDistance(optional UTBRPlayerManager playerManager, optional float height)
{
    local float speed, dist, gravMult, heightMult;
       
    speed = GetShootSpeed();
    
    //distance ball will travel with default ball speed of 1300 in reg br gravity of -420.
    //found ball travels about about twice as far in low grav(-100).
    dist = 1300;

    gravMult = GetTravelGravityMultiplier();
   
    //distance increases somewhat exponentially as height increases, especially in low grav
       
    if (UTBRGame(WorldInfo.Game).IsLowGravity())
    {
        heightMult = height / 250.0;
    }
    else
    {
        heightMult = height / 500.0;        
    }

    //heightMult = 3 + (gravMult * 2.0 * heightMult);    
    heightMult = 4 * gravMult;
    
    if (height <= 150)
    {
        //at lower height the increase in distance is not so great
        //heightMult *= ((height / 150) * (height / 150));
    }
    
    if (height <= 0)
    {
        heightMult = 0;
    }   

    dist = dist + (height * heightMult);    
    
    dist = dist * (speed / 1300) * gravMult;
    
    return dist;
}

//return multiplier according to gravity for ball travel distance calculations
function float GetTravelGravityMultiplier()
{
    return UTBRGame(WorldInfo.Game).GetTravelGravityMultiplier();
}

//throw ball starting at startLocation in Direction
function Throw(vector startLocation, vector Direction)
{
    local float specialZ, speed;

    speed = GetShootSpeed();   
    specialZ = holderBallLauncher.SpecialTossZ;
    holderBallLauncher.SpecialTossZ = 0;
    HolderPlayerManager.AimTime = 0;
        
    DetachFromHolder(DETACH_THROW);
    
    SetLocation(startLocation); 
    SetRotation(rotator(Direction));

    Velocity = Direction * speed;
    
    //should not set tossz to anything but zero, as in low grav it throws off aim
    Velocity.Z += TossZ + specialZ;
    
    //adjust flight path up a little according to gravity.
    //this helps accuracy when shooting ball, especially when shooting a goal.
    //in regular gravity of -420 this results in 250.    
    Velocity.Z += (WorldInfo.WorldGravityZ * 0.595238 * -1);
    
    HasHitWall = false;               
    
    //don't mess with acceleration. seems to be a problem in ut. acceleration can't be readjusted in HitWall and
    //so the ball goes crazy when it hits a wall with acceleration.
    //currently in standard game, objects which use acceleration are PHYS_PROTECTILE objects which don't change direction on HitWall.
    //it'd be nice to have acceleration to be able to make high shots go further if we wanted to.
    //however, the gaming community voted to keep default ball speed down to what ut2004 had, they do not want a long 
    //throw. generally, they prefered a slower throw. It's easier to boost, catch etc, and easier to manage the 
    //ball on smaller maps.
    //one very interesting reason was that on some maps you had to shoot and then boost the ball into the goal, 
    //and with a longer throw distance that would no longer be the case.
    //to accomodate the desire to perhaps have a longer throw on larger maps, we have the BallSpeed ini setting.
    //Acceleration = AccelRate * Normal(Velocity);

        
    if( PassTarget != None )
    {
        PassSeeking = true;
        ClientPassSeek();

        if (PlayerController(GetController(PassTarget)) != none)
        {
            PlayerController(GetController(PassTarget)).ReceiveLocalizedMessage( class'UTBRHUDMessage', 3 );            
        }
    }
    
    ClientDetachFromHolder();
    ChooseBoostDefender(1 - LastHolder.GetTeamNum());
}

//called on server and on also on client to make sure things get replicated ok
simulated function ClientDetachFromHolder()
{
    //seen strange things in net play like when shooting, ball just teleports around without 
    //any visible movement, which is caused by physics not getting replicated to client.
        
    if (PassTarget == none)
    {
        SetPhysics(PHYS_FALLING);
    }
    
    bCollideWorld = true;
    SetCollision(true, true);   
    SetOwner(none);  
    SetOwnerNoSee(false);

    //adjust light to center of ball so it shines ok on floor when ball is on floor    
    BallLight.SetTranslation(vect(0,0,0));   
}

//called on server and on client when ball is thrown by ball launcher
simulated function ClientThrown()
{
	flightTrail = Spawn(class'UTEmitter', self,, Location, Rotation);
	flightTrail.SetBase(self);
	flightTrail.SetRelativeLocation(vect(0,0,0));	
	
	if (GetTeamNum() == 0)
	{
	    flightTrail.SetTemplate(ParticleSystem'BombingRunGraphics.BallTrailRed', true);
	}
	else
	{
	    flightTrail.SetTemplate(ParticleSystem'BombingRunGraphics.BallTrailBlue', true);			
    }
}

//return velocity of holder, even if in vehicle.
//if in vehicle, Holder.Velocity is 0, so must use vehicle velocity.
function vector HolderVelocity()
{
    if ((HolderController != none) && (Vehicle(HolderController.Pawn) != none))
    {
        return HolderController.Pawn.Velocity;
    }

    if (Holder != none)
    {
        return Holder.Velocity;
    }


    return vect(0, 0, 0);
}

//detach ball from current holder.
//could be detached by dying, throwing, scoring, etc.
function DetachFromHolder(int detachType, optional Controller Killer)
{
    bForceNetUpdate = True;

    if (DroppedSound != None)
    {
        PlaySound(DroppedSound);
    }   

    DetachCarrierEffect(GetTeamNum());

    SetReturnTimer(true);    
    Velocity = HolderVelocity();
    isDropped = true;
    bHome = false;      
    ClearTimer('BallTimer');
    OldHolder = Holder;
    UTGameReplicationInfo(WorldInfo.GRI).SetFlagDown(GetTeamNum());
    WorldInfo.GRI.bForceNetUpdate = TRUE;   
    BaseBoneName = '';
    BaseSkelComponent = None;
    SetBase(None);
    bForceNetUpdate = TRUE;
    HolderPlayerManager.LastJumpTime = 0;

    OldHolderVehicle = none;
    if (Vehicle(HolderController.Pawn) != none)
    {
        OldHolderVehicle = Vehicle(HolderController.Pawn);
    }
    
    if (UTBot(HolderController) != none)
    {
        if (! bool(UTBRGame(WorldInfo.Game).Settings.AllowBallCarrierWeapons))
        {
            UTBot(HolderController).ImpactJumpZ = holderPlayerManager.oldImpactJumpZ;
    
            UTBot(HolderController).bHasTranslocator = holderPlayerManager.OldHasTranslocator;
            if (UTBot(HolderController).bHasTranslocator)
            {
                UTBot(HolderController).bAllowedToTranslocate = true;
                UTBot(HolderController).NextTranslocTime = WorldInfo.TimeSeconds - 1;             
            }
        }
    }   

    if (detachType != DETACH_THROW)
    {
        ShotByBallLauncher = false;
        
        //must release passtarget before calling ClientDetachFromHolder because it checks PassTarget var
        ReleasePassTarget();
          
        holderBallLauncher.OnBallDetached();
        
        //needed for when feigning death, weapon switching is disabled, but must
        //force ball launcher to get switched out
        Holder.Weapon = none;           
    }
        
    if (detachType == DETACH_DROP)
    {        
        if ( (OldHolder != None) && (OldHolder.health > 0) )
        {
            Velocity = 0.5 * HolderVelocity();
            if ( Holder.Health > 0 )
                Velocity += 300*vector(Holder.Rotation) + 100 * (0.5 + FRand()) * VRand();
        }
        Velocity.Z = 250;
        if ( PhysicsVolume.bWaterVolume )
            Velocity *= 0.5;
                
        if ( Killer != None )
        {
            LogDropped(Killer);
        }
        else
        {
            LogDropped(HolderController);
        }       

        //for DETACH_THROW, Throw() will call this.
        //for DETACH_SEND_HOME, not needed to call, as SendHome() will call ClientReturnedHome() to set the proper things.
        ClientDetachFromHolder();       
    }
    
    isThrown = (detachType == DETACH_THROW);
    if (isThrown)
    {
       ClientThrown();
    }
    
    GotoState('Dropped');

    ClearHolder();
    
	Holder = None;
	HolderPRI = None;
    HolderController = none;
    holderBallLauncher = none;
            
    UTBRGame(WorldInfo.Game).SetPlayerObjectives(false);
   
    CheckTouching();    
    
    //CheckFit();
    //CheckPain();      
}

state Dropped
{
	function SendHome(Controller Returner)
	{
	  	Global.SendHome(Returner);
	}

	function Timer()
	{
    }
    	
    function LogTaken(Controller EventInstigator)
    {
        Global.LogTaken(EventInstigator);
    }
        
    singular function PhysicsVolumeChange( PhysicsVolume NewVolume )
    {
        CheckPain();
    }
    
	function bool FlagUse(Controller C)
	{
	   return false;
    }    

	event Touch(Actor Other, PrimitiveComponent OtherComp, vector HitLocation, vector HitNormal)
	{
		global.Touch(Other, OtherComp, HitLocation, HitNormal);
	}
	
    event TakeDamage(int Damage, Controller EventInstigator, vector HitLocation, vector Momentum, class<DamageType> DamageType, optional TraceHitInfo HitInfo, optional Actor DamageCauser)
    {
        global.TakeDamage(Damage, EventInstigator, HitLocation, Momentum, DamageType, HitInfo, DamageCauser);
    }   

    singular event BaseChange()
    {
        /*
        if (Pawn(Base) != None)
        {
            Velocity = 100.0 * VRand();
            Velocity.Z += 200.0;
            SetPhysics(PHYS_Falling);
        }
        */
    }   
        
    function BeginState(Name PreviousStateName)
    {
                ClearTimer();
    }

    function EndState(Name NextStateName)
    {
    }
}

function LogDropped(Controller EventInstigator)
{
    GetKismetEventObjective().TriggerFlagEvent('Dropped', EventInstigator);
    if ( bLastSecondSave && (EventInstigator != HolderController) )
    {
        BroadcastLocalizedMessage(class'UTLastSecondMessage', 1, HolderPRI, None, EventInstigator.PlayerReplicationInfo.Team);
        if ( UTPlayerReplicationInfo(EventInstigator.PlayerReplicationInfo) != None )
        {
            UTPlayerReplicationInfo(EventInstigator.PlayerReplicationInfo).IncrementEventStat('EVENT_LASTSECONDSAVE');
        }
    }
    else
    {
        BroadcastDroppedMessage(EventInstigator);
    }
    bLastSecondSave = false;
}

//release the current pass lock
simulated function ReleasePassTarget()
{
    if (IsTimerActive('PassSeekTimer'))
    {
        ClearTimer('PassSeekTimer');
        
        Velocity = vect(0,0,0);
        Acceleration = vect(0,0,0);         
            
        //there must be a change in physiscs for this to work.
        //here we change from projectile to falling, otherwise
        //ball just sticks to wall and bounces endlessly.
        SetPhysics(PHYS_Falling);       
    }
    
    if ((Role == Role_Authority) && (PassTarget != none) && (LastHolder != none) && 
        (Holder != PassTarget) && (PlayerController(GetController(LastHolder)) != none))
    {
       PlayerController(GetController(LastHolder)).ClientPlaySound(LockLostSound);
    }

    PassSeeking = false;
    PassTarget = none;
}

//set the timer which will automatically send ball home after a certain time
//timer is turned off by things like shooting at the ball, and is set when
//ball is not at it's base and sitting idle.
function SetReturnTimer(bool TurnOn)
{
    if (UTBRGame(WorldInfo.Game).Test_NoBallReset)
    {
        return;
    }
    
    if (TurnOn)
    {
        if (! IsTimerActive('BallReturnTimer'))
        {
            SetTimer(MaxDropTime, false, 'BallReturnTimer');
        }
    }
    else
    {
        ClearTimer('BallReturnTimer');
    }
}

//time to send ball home if not at base and sitting idle for a certain period of time
function BallReturnTimer()
{
    AutoSendHome();
}

simulated event Bump( Actor Other, PrimitiveComponent OtherComp, Vector HitNormal )
{
    if ((! isDropped) && (! bHome))
    {
        return;
    }
        
    BallTouched(Other);
}
                

//called to boost the ball when it's shot at
simulated event TakeDamage(int DamageAmount, Controller EventInstigator, vector HitLocation, vector Momentum, class<DamageType> DamageType, optional TraceHitInfo HitInfo, optional Actor DamageCauser)
{
    if (! isDropped)
    {
        return;
    }
        
    if (Momentum != Vect(0,0,0))
    {
        SetPhysics(PHYS_Falling);
        Velocity += Momentum/Mass;           
               
        if ( WorldInfo.NetMode != NM_DedicatedServer )
        {
            PlaySound(ImpactSound, true);
        }
        
        if (EventInstigator == TryingToDeflect)
        {
            //once bot hits ball, make it stop
            TryingToDeflect = none;
        }           
    }   
}

//return true if ball has a pass target set.
//can't just test for PassTarget != none because it can be set to none client side via replication.
simulated function bool PassTargetActive()
{
    return IsTimerActive('PassSeekTimer') || (physics == PHYS_PROJECTILE) || (PassTarget != none);
}

simulated event HitWall(vector HitNormal, Actor Wall, PrimitiveComponent WallComp)
{ 
    local bool bHasHitWall;     


    //simulated events aren't getting called in simulated states. ut bug. so, must have as global event.
    //HitWall is continuously called while ball is sitting on floor.
    //touched event only gets fired once even if actor is continuously touching.
       
    bHasHitWall = HasHitWall;
    HasHitWall = true;
    
    if ((! isDropped) || (OldHolder == Wall) || (! Wall.bBlockActors))
    {
        return;
    }
    
    //note on BlockingVolumes. ball will bounce off them but projectiles
    //will not. projectiles have a bSwitchToZeroCollision setting to turn off
    //collsion with these volumes but it's hard coded native and only available
    //for projectiles. other things like pawns will also bounce off these volumes.

    if (Wall != OldHolderVehicle)
    {
       //by only resetting OldHolder when hits something other than OldHolder or OldHolder's vehicle, we stop
       //ball from being repicked up immediately when dropped
       OldHolder = none;
    }
    
    if (IsTimerActive('PassSeekTimer'))
    {
        ReleasePasstarget();
    
        return;
    }
    
    if (ValidHolder(Wall))
    {
        BallTouched(Wall);
        return;    
    }
           
    //bounce ball off wall
    
    Velocity = Elasticity*(( Velocity dot HitNormal ) * HitNormal * (-2.0) + Velocity);
    
    //give it a little extra bounce the first bounce in higher gravity.
    //in low grav(-100) it has plenty of bounce.
    if ((! bHasHitWall) && (WorldInfo.WorldGravityZ < -200))
    {
        //in regular gravity of -420 this results in 200
        Velocity.Z += 0.476190 * WorldInfo.WorldGravityZ * -1;
    }
    
    SetPhysics(PHYS_Falling);                         
           
    Acceleration = vect(0,0,0);     
         
    if (VSize(Velocity) > 20)
    {                            
        if ( WorldInfo.NetMode != NM_DedicatedServer )
        {
            PlaySound(ImpactSound, true);
        }               
    }
    else
    {
        Resetting = false;    
    }
}   

//called to navigate ball while seeking a pass lock target.
//with pass locking the ball holder alt clicks to lock on a team mate and then they fire
//to make the ball seek out that player.
simulated function PassSeekTimer()
{
    local vector Dir;
    local float CollisionRadius, CollisionHeight;

    if ((PassTarget == none) || (Holder != none) || (! isDropped))
    {
        return;
    }
    
    PassTarget.GetBoundingCylinder(CollisionRadius, CollisionHeight);               
    Dir = Normal((PassTarget.Location + (vect(0,0,1)*CollisionHeight*0.5)) - Location);

    Acceleration = 1.25 * float(UTBRGame(WorldInfo.Game).Settings.BallSpeed) * Dir;

    if ( VSize(Location - PassTarget.Location) < 250 )
    {
        Velocity = Velocity + 10 * Acceleration;

        if ( (Dir Dot Normal(Velocity)) < 0.9 )
            Acceleration *= 0.4;
    }
    SetRotation(rotator(Acceleration));
    
    Velocity = Velocity + 10 * Acceleration;
}

function CheckTouching()
{
    local int i;

    for ( i=0; i<Touching.Length; i++ )
    {
        BallTouched(Touching[i]);
    }
}
        
function BallTouched(Actor other)
{
    if (! Enabled)
    {
        return;
    }
              
    if (ValidHolder(other))
    {
        SetHolder(GetController(Pawn(other)));
    }      
}

//called when dropped ball is touched. check and see if other can take the ball.
function bool ValidHolder(Actor other)
{
    local Pawn p;
    local UTPawn UTP;
    local UTBRPlayerManager playerManager;
       
    if ((Pawn(other) != none) && (UTBot(GetController(pawn(other))) != none) &&
        UTBRGame(WorldInfo.Game).Test_NoBallPickup)
    {
        return false;
    }
    
    if ((Holder != none) && (Holder == other))
    {
        return true;
    }
    
    p = Pawn(other);
    
    if (p == none)
    {
        return false;
    }
       
    if ((UTVehicle(p.base) != None) && (! UTVehicle(p.base).bCanCarryFlag))
    {
        return false;
    }
    
    if ((UTVehicle(p) != None) && (! UTVehicle(p).bCanCarryFlag))
    {
        return false;
    }
        
    if (UTVehicle(p) != none)
    {
        if (UTVehicle(p).Health <= 0)
        {
            return false;
        }
        
        p = UTVehicle(p).Driver;
    }
            
    if ( p == None || p.Health <= 0 || p == OldHolder || (Holder != none) )
    {
        return false;
    }
    
    if ((! p.bCanPickupInventory) && (p.DrivenVehicle == none))
    {
        return false;
    }
         
    if ((! p.IsPlayerPawn()) && (p.DrivenVehicle == none))
    {
        return false;
    }     
       
    // feigning death pawns can't pick up flags
    UTP = UTPawn(Other);
    if (UTP != none)
    {       
        if (UTP.IsInState('FeigningDeath'))
        {
            return false;
        }

        if (! bool(UTBRGame(WorldInfo.Game).Settings.AllowBallCarrierWeapons))
        {
            playerManager = UTBRGame(WorldInfo.Game).GetPlayerManager(UTP.PlayerReplicationInfo);

            if ((UTWeapon(UTP.Weapon) != none) && 
                (! UTWeapon(UTP.Weapon).AllowSwitchTo(playerManager.BallLauncher)))
            {
                if ((playerManager != none) && ((WorldInfo.TimeSeconds - playerManager.WeaponDisallowBallPickupTime) > 5))
                {
                    playerManager.WeaponDisallowBallPickupTime = WorldInfo.TimeSeconds;
                    PlayerController(UTP.Controller).ClientMessage("You can not carry the orb while holding this weapon");
                }

                return false;      
            }
        }
    }
    
    return true;
}

singular event Touch( Actor Other, PrimitiveComponent OtherComp, vector HitLocation, vector HitNormal )
{
    if ((! isDropped) && (! bHome))
    {
        return;
    }
       
    if (ValidHolder(Other))
    {
        BallTouched(Other);       
    }
    else
    {
        /*
           Usually HitWall is called for collision but for certain actors Touch is called.
           alas, must hard code the kind of actors we want to pass this check.
           Some actors like ForcedDirVolume have bBlockActors set but they don't
           actually block and you must check StopsProjectile.
           there is a StopsProjectile() in actor but it requires Projectile parameter.
           thought about making a dummy projectile to pass but it's too much work to make
           a non functional projectile class and also some things like teleporters actually operate
           on the passed projectile.
           Usually an actor stops a projectile if bBlockActors is true.
           Of the list of actors which overrides StopsProjectile, the relevant ones are
           included below.
        */
        if ((UTVehicle(Other) != none) || (UTWalkerBody(Other) != none))
        {
            HitWall(HitNormal, Other, OtherComp);
        }      
    }
}

simulated function SetEnabled(bool bEnable)
{    
    SetHidden(! bEnable);
    Enabled = bEnable;
    
    BallLight.SetEnabled(bEnable);
}

//switch off certain visuals when in first person mode.
//people found the big player rings annoying in first person.   
simulated function SetOwnerNoSee(bool cantSee)
{
    BallEffectComp.SetOwnerNoSee(cantSee);
    BallMesh.SetOwnerNoSee(cantSee);
    CarrierEffects[0].ParticleSystemComponent.SetOwnerNoSee(cantSee);
    CarrierEffects[1].ParticleSystemComponent.SetOwnerNoSee(cantSee);
}


replication
{
    if (Role == ROLE_Authority)
        TeamIndex, isDropped, PassTarget, LastHolder, PassSeeking, Enabled, CarrierEffects, isThrown;
}

defaultproperties
{
   Begin Object Class=DynamicLightEnvironmentComponent Name=BallLightEnvironment ObjName=BallLightEnvironment Archetype=DynamicLightEnvironmentComponent'Engine.Default__DynamicLightEnvironmentComponent'
      bDynamic=True
      bCastShadows=False        
      AmbientGlow=(R=4.000000,G=4.000000,B=1.000000,A=1.000000)
      Name="BallLightEnvironment"
      ObjectArchetype=DynamicLightEnvironmentComponent'Engine.Default__DynamicLightEnvironmentComponent'
   End Object
   BallSurfaceLight = BallLightEnvironment

   //the blue doesn't shine as brite as red, so give blue a little more
   BallSurfaceColors(0)=(B=0,G=1,R=4,A=0)
   BallSurfaceColors(1)=(B=5,G=1,R=0,A=0)
   BallSurfaceColors(2)=(B=1,G=4,R=4,A=0)  

   /*
      gives the ball a glow which shines on walls and etc.
      used to use two lights on opposite ends of ball to light ball evenly, and their radius was 250.
      had to use only one light at smaller radius because framerate usage was too great.
      dynamic lights are very expensive. framerate savings is about 3 per 25 radius.
      100 looks good and we would not want to go any higher due to framerate cost.
      could go down to 75 or 50 but doesn't look as nice.
      because of smaller radius light has trouble showing on floor when ball is held, 
      so when held we adjust light to be down below the ball so that it can still
      shine on floor.
      we now use BallSurfaceLight to give the ball surface shine and color.
      there is an uneven shine on ball due to using a single pointlight instead of two,
      and we'll have to live with that.
   */     
   Begin Object Class=PointLightComponent Name=BallLightComponent0 ObjName=BallLightComponent0 Archetype=PointLightComponent'Engine.Default__PointLightComponent'
      Radius=100.000000
      Brightness=5.0
      LightColor=(B=64,G=255,R=255,A=0)       
      Name="BallLightComponent0"
      Translation=(X=0.000000,Y=0.000000,Z=0)
      ObjectArchetype=PointLightComponent'Engine.Default__PointLightComponent'
      CastDynamicShadows=False
      CastShadows=false
      CastStaticShadows=false
      bCastCompositeShadow=false
   End Object
   BallLight=BallLightComponent0
       
          
   Begin Object  Class=StaticMeshComponent Name=BMesh ObjName=BMesh Archetype=StaticMeshComponent'Engine.Default__StaticMeshComponent'
      StaticMesh=StaticMesh'PICKUPS.PowerCell.Mesh.S_Pickups_PowerCell_Cell01'    
      LightEnvironment=BallLightEnvironment
      bUseAsOccluder=False
      CastShadow=False
      Scale=1.350000
      Name="BMesh"
      ObjectArchetype=StaticMeshComponent'Engine.Default__StaticMeshComponent'
      CollideActors=True
      BlockActors=True
      BlockRigidBody=True      
   End Object
   BallMaterials(0)=Material'PICKUPS.PowerCell.Materials.M_Pickups_Orb_Red'
   BallMaterials(1)=Material'PICKUPS.PowerCell.Materials.M_Pickups_Orb_Blue'   
   BallMaterials(2)=Material'BombingRunGraphics.Ball_Shell_Yellow'
   BallMesh=BMesh
   
   BallColors(0)=(B=0,G=0,R=255,A=0)
   BallColors(1)=(B=255,G=0,R=0,A=0)
   BallColors(2)=(B=64,G=255,R=255,A=0)
         
   LastLocationPingTime=-100.000000
   bHome=True
   bUseTeamColorForIcon=False
   
   //note: it would be better to have this is UTBRBallBase, but alas it's defined and used in UTCarriedObject
   HomeBaseOffset=(X=0.000000,Y=0.000000,Z=50.000000)
      
   MapSize=0.300000

   IconCoords=(U=843.000000,V=0.000000,UL=50.000000,VL=48.000000)   
      
   Begin Object  Name=CollisionCylinder ObjName=CollisionCylinder Archetype=CylinderComponent'UTGame.Default__UTCarriedObject:CollisionCylinder'
      CollisionRadius=26.5
	  CollisionHeight=22.5     
      ObjectArchetype=CylinderComponent'UTGame.Default__UTCarriedObject:CollisionCylinder'
      CollideActors=True
      BlockActors=True
      BlockRigidBody=True       
   End Object
   
   bHardAttach=True
   
   //using twice the priority of a pawn. ball is a center piece of game and needs to
   //update a lot and there is only one ball so make it very high priority.
   //we tried to have ball update like a projectile (once upon being fired and then on every wall bounce)
   //and people noticed that sometimes the ball would jerk and jump around sometimes and 
   //not truely be where you saw it at so the below way is better. ball may still shake a little in travel
   //on slower rigs but we'll have to live with that. 
   NetPriority=6.000000
   NetUpdateFrequency = 100;
   
   CollisionComponent=CollisionCylinder
   Name="Default__UTBRBall"
   ObjectArchetype=UTOnslaughtFlag'UTGame.Default__UTOnslaughtFlag'   
     
   PickupSound=SoundCue'A_Gameplay.CTF.Cue.A_Gameplay_CTF_FlagPickedUp01Cue'
   DroppedSound=SoundCue'A_Gameplay.CTF.Cue.A_Gameplay_CTF_FlagDropped01Cue'
   ReturnedSound=SoundCue'A_Gameplay.CTF.Cue.A_Gameplay_CTF_FlagReturn_Cue'
   
   
   CylinderComponent=CollisionCylinder
   MessageClass=Class'BombingRun.UTBRBallMessage'
   
   BallEffects(0)=ParticleSystem'BombingRunGraphics.Ball_Ring_Red'
   BallEffects(1)=ParticleSystem'BombingRunGraphics.Ball_Ring_Blue'
   BallEffects(2)=ParticleSystem'BombingRunGraphics.Ball_Ring_Yellow'
            
   Begin Object Class=ParticleSystemComponent Name=EffectComp ObjName=EffectComp Archetype=ParticleSystemComponent'Engine.Default__ParticleSystemComponent'
      SecondsBeforeInactive=1.000000
      Name="EffectComp"
      ObjectArchetype=ParticleSystemComponent'Engine.Default__ParticleSystemComponent'
   End Object
   BallEffectComp=EffectComp
   
   Components(0)=CollisionCylinder
   Components(1)=BallLightComponent0
   Components(2)=BallLightEnvironment
   Components(3)=BMesh
   Components(4)=EffectComp     
     
   bBounce=True
   bBlockActors=True
   bCollideWhenPlacing=True
   bCollideActors=True
   bCollideWorld=True           // Collides with the world.
   BlockRigidBody=True 
   Mass=100.0
   Elasticity=0.4   
   bMovable=True    
   Physics=PHYS_None
   ImpactSound=SoundCue'A_Weapon_RocketLauncher.Cue.A_Weapon_RL_GrenadeFloor_Cue'   
   bCanBeDamaged=true
   SeekAccum=0.0   
   SeekInterval=0.05   
   ThrowerTouchDelay=1.f   
   LockAcquiredSound=SoundCue'A_Weapon_RocketLauncher.Cue.A_Weapon_RL_SeekLock_Cue'
   LockLostSound=SoundCue'A_Weapon_RocketLauncher.Cue.A_Weapon_RL_SeekLost_Cue'   
   TossZ=0
   AccelRate=0
   RemoteRole=ROLE_SimulatedProxy
   bReplicateInstigator=True
   bUpdateSimulatedPosition=True;
   CustomGravityScaling=0.1
   TeamIndex=-1
   bNoDelete=false
   GameObjOffset3P=(X=-30.000000,Y=0.000000,Z=50.000000)
   GameObjOffset1P=(X=-30.000000,Y=0.000000,Z=50.000000)
   Enabled = true
   CameraViewDistance=400.000000
   bCollideComplex=True
   bCanTeleport = true   
}
